pico-8 cartridge // http://www.pico-8.com
version 42
__lua__
-- marballs
-- by lucatron

--general constants
sw = 24 --main tile sprite width
sh = sw/2 --height
tc = (sqrt(2)*128)/sw + 1 --tile capacity along diagonal, for checking if tiles are onscreen
cntdwn = 25 --countdown length
t = 0 --stage frame, resets on level start
tg = 0 --global frame count

--character/player
c_x = 0 c_y = 0 c_z = 0 --pos (pixel)
c_mx = 0 c_my = 0 --pos (map)
c_vx = 0 c_vy = 0 --velocity
c_vz = 0
c_vmin = 0.01 c_vmax = 3 --max speeds
c_acc = 0.43 c_deacc = 0.92 --acceleration
c_av = 3 --jump speed
c_g = 0.3 --gravity acceleartion
c_aacc = 0.12 c_adeacc = 0.99 --air acceleration

--map
m_x = 0 --x tile of map
m_w = 32 --width/height
m_h = 32
m_tc = {{3,11}, --tile colours
        {13,6}, --for each lvl
        {4,15}} 
m_name = {"rolling", --level names
          "jumping",
          "routing"}
m_hs = {999,999,999} --highscores
m_dhs = {7.89,17.76,16.36} --dev highscore
m_nhs = false --if new highscore
lvls = 3 --number of levels

--g_paused menu
p_c = 0 --cursor
p_xo = 26 --x/y offset
p_yo = 60
p_vo = 12 --vertical offset
p_to = (64-p_yo)/2 --text offset
p_sh = 1 --border width

--sound channels
sd_menu = 0
sd_roll = 0
sd_jump = -1
sd_music = 2 + 4 + 8
sd_songs = {11,1,19} --songs for each level

---boot animation logo
--table turned out to be less
--effecient so was not used here
l_pww = 45 --pinwheel width
--actually quite cpu intensive
--so pww>45 starts getting laggy
l_pws = 180 --pinwheel slowness
l_pwy = 30 --pinwheel y-pos
l_lsm = 2 --logo size multiple
l_ly = 88 --logo y-pos
l_len = 115 --length of animation
l_wait = 30 --length of blackscreen wait

--title constants/variables
t_sm = 3 --title width multiplye
t_xs = 1 --x bg speed
t_ys = 0.2 --y bg sped
t_sy = 0 --sin motion of y
t_w = 42 --drawn board width/2
t_res = 1 --resolution of boar

--transition
tr_go = false --if transitioning
tr_t = 0 --g_timer
tr_l = 32 --length of fade
tr_p = 10 --length of pause
tr_to = -1 --where to fade to
tr_tog = 0 --to game
tr_tos = 1 --to scoreboard
tr_tot = 2 --to menu

--game conditions
g_lvl_end = false --if beaten level
g_lvl_death = false
g_paused = false
lvl = 0 --level number
g_timer = 0 --g_timer
g_flp = false --flipper

game_state = 0 --variables
gs_boot = 0 --booting
gs_title = 1 --main menu
gs_game = 2 --ingame
gs_score = 3 --scorebaord


-----------------
--debug settings
debug = false --turn on/off

if debug then
 l_len = 2
 l_wait = 2
 tr_l = 4
 tr_p = 4
 cntdwn = 4
 for i=0,lvls-1 do
  mset(2+m_w*i,0,17)
 end
end


---------------------
--general functions--

--return sign of n
function sign(n)
 if (n < 0) return -1
 return 1
end

--round float n to d places
function round(n,d)
 return flr(n*10^d)/10^d
end

--print with simple "3d" shadow
function prn(st,x,y,col)
 print(st,x,y+1,5)
 print(st,x,y,col)
end


---------------------
--loading functions--

--change level
function load_level(newlvl)
 game_state = gs_game
 lvl = newlvl
 m_x = m_w*lvl --update map pos
 reset_level()
 music(-1)
 music(sd_songs[lvl+1],tr_l*40,sd_music)
end


function load_title()
 game_state = gs_title
 t = 0
 music(-1)
 music(15,2000)
end

function load_scoreboard()
 game_state = gs_score
 music(-1)
 music(11,tr_l*40)
end

--initialize ball
function init_ball()
 c_x = 0
 c_y = sw/2
 c_z = 0
 c_mx = m_x
 c_my = 0
 c_vx = 0
 c_vy = 0
 c_vz = 0
end

--restart the level
function reset_level()
 if g_lvl_end then
  music(-1)
  music(sd_songs[lvl+1],tr_l*40,sd_music)
 end

 init_ball()
 g_paused  = false
 g_lvl_end = false
 g_lvl_death = false
 g_flp = false
 g_timer = 0
 t = 0
 m_nhs = false
 p_c = 0
end


function _init()
 game_state = gs_boot
end


---------------------
--drawing functions--


--bootup logo
function draw_logo() 
--stick
 sspr(64,8,8,8,56,2+l_pwy,16,50)
 
--make angle
--note polynomial of t, which accelerates it
 a = (t+0.04*t^2)%l_pws/(l_pws-1)
 
--precalculate trig functions
 cosa = cos(a)
 sina = sin(a)
 
--iterate over square
--centered on (0,0)
 for x=-l_pww/2,l_pww/2-1 do
 for y=-l_pww/2,l_pww/2-1 do
  --rotate x,y coords around (0,0)
  x2 = cosa*x - sina*y
  y2 = sina*x + cosa*y
  
  --if in square
  if (abs(x2) <= l_pww/2 and
      abs(y2) <= l_pww/2) then
   --get related pixel colour
   col = sget(x2/l_pww*8  + 68,
              y2/l_pww*8 + 4)
   if col~=0 then
    pset(x+64, y+l_pwy,col)
   end
  end
 end
 end
 
 --the white axle
 circfill(64,l_pwy,3,7)
 
 --company name
 sspr(72,0,32,16,
      64-16*l_lsm,l_ly,
      32*l_lsm,16*l_lsm)
end

--title screen
function draw_title()
 --draw perspective map thing
 --precalculate stuff
 sa = sin(t/120%1) ca = cos(t/120%1) --trig stuff
 p = 0.45 + 0.12*sa --perspective ratio (1=vertical,0=horizontal)
 ip = 1/p --inverse of p
 maxw = t_w*sqrt(2) --max width (that a pixel could be draw)
 maxh = t_w*sqrt(2)*p --max height
 
 for x=-maxw,maxw,t_res do
  for y=-maxh,maxh,t_res do
   --rotate coords around 0,0
   rx = ca*x - sa*y*ip
   ry = sa*x + ca*y*ip
   --if in square
   if abs(rx) <= t_w and
      abs(ry) <= t_w then
    --get realtive tile
    tilx = flr((rx+t_w)/(t_w/16)+m_w)
    tily = flr((ry+t_w)/(t_w/16))
    til = mget(tilx,tily)
    
    --if an actual tile
    if til!=0 then
     --get checkerboard colour
     col = 11
     if ((tilx+tily)%2 == 0) col = 3
     px = x + 64
     py = y + 70
     --top
     rectfill(px,py,
             px+t_res,py+t_res,
             col)
     --shadow
     rectfill(px,py+5,
              px+t_res,py+t_res,
              1)
    end
   end
  end
 end
 
 --title name
 pal()
 sspr(8,24,40,8,
      64-20*t_sm,10,
      40*t_sm,8*t_sm)
 
 --press start text
 prn("press z/x to begin",
       28,110,7)
end


function draw_transition()
 --first half of wipe
 if tr_t < tr_l then
  y = tr_t/(tr_l)*127
  rectfill(0,0,127,y,0)
 --blackscreen pause
 elseif tr_t < tr_l + tr_p then
  rectfill(0,0,127,127,0)
 --second halfe wipe
 else
  y = (tr_t-tr_l-tr_p)/(tr_l)*127
  rectfill(0,y,127,127,0)
 end
end

--ingame background
function draw_bg()
 --circle radius
 rad = 12+8*cos(tg/500%1)
 --distance between each circle
 step = rad*3
 
 --note iteration starts at 64
 --to give centred zoom
 for x=64,127+rad,step do
  for y=64,127+rad,step do
   circfill(x,y,rad,1)
   circfill(128-x,128-y,rad,1)
   circfill(128-x,y,rad,1)
   circfill(x,128-y,rad,1)
  end
 end
end

--draw level tiles
function draw_iso_map()
 --iterate over map tiles
 for y=0,m_h-1 do
 for x=0,m_w-1 do
 
  --if a tile
  --and if onscreen (fast approx)
  if (mget(x+m_x,y) ~= 0 and
      abs(c_mx-x-m_x) < tc and
      abs(c_my-y) < tc) then
   --get isometric coords
   rx = m_w-1 + x - y
   ry = (x + y)/2
   --get pixel coords, which
   --centres map on player
   px = rx*sh - m_h*sh + 64 - c_x
   py = ry*sh + 64 - c_y/2
  
   --get tile type
   til = mget(x+m_x,y)
   pal()
   
   --select tile coluors
   --normal tile
   if (til == 1) then
    col = m_tc[lvl+1][1]
    if ((x+y)%2==0) col = m_tc[lvl+1][2]
    pal(2,col)
    pal(3,col)
   --blue flipper
   elseif (til == 5) then
    pal(2,12)
    pal(3,12)
    if (g_flp) palt(3,true)
   --red flipper
   elseif (til == 21) then
    pal(2,9)
    pal(3,9)
    if (not g_flp) palt(3,true)
   --b/w checker finish
   elseif (til == 17) then
    pal(2,6)
    pal(3,7)
   end
   
   --draw the sprite
   sspr(16,0,sw,sh,px,py)
  end
 end
 end
end


function draw_player()
 --shadow
 pal(1,0)
 --if small or big shadow
 n = 37
 if (c_z>6) n = 36
 spr(n,60,60)
 pal()
 --ball
 spr(32,60,56-c_z)
 --shimmer, some magic numbers
 --here to get a feel
 spr(33 + 0.15*(c_x+0.6*c_y)%3,60,56-c_z)
end

--draw the menu
function draw_g_paused()
 rectfill(p_xo-p_sh,p_yo+p_vo-p_sh,
  128-p_xo+p_sh,128-p_yo+p_vo+p_sh, 0)
 rectfill(p_xo,p_yo+p_vo,
  128-p_xo,128-p_yo+p_vo, 13)
 
 ty = p_yo+p_to+p_vo
 
 if g_lvl_death then
  prn("restart",50,ty,10)
 else
  --option 1
  col = 7
  if (p_c==0) col = 10
  s="resume"
  if (g_lvl_end) s="next lvl"
  prn(s,p_xo+p_to,ty,col)
  
  --option 2
  col = 7
  if (p_c==1) col = 10
  prn("restart",100-p_xo,ty,col)
 end
end

--level timer
function draw_timer()
 sec = g_timer/30
 str = "time: "..flr(sec/60)
     ..":"..round(sec%60,2)
 if m_nhs then
  str = str.."  new best time!"
 end
 --flash on level end
 col = 7
 if g_lvl_end and t%20 < 10 then
  col = 6
 end
 
 prn(str,2,3,col)
end

--starting countdown g_timer
function draw_countdown()
 cd_sm = 2 --size multiplyer
 
 --get y of spr. this is... yeah...
 if (t%cntdwn < cntdwn/3) then
  y = -8*cd_sm+t%cntdwn/cntdwn*50*3
 else 
  y = -8*cd_sm+cntdwn/3%cntdwn/cntdwn*50*3
 end
 
 if t <= cntdwn then --ready?
  sspr(48,16,32,8,
       64-cd_sm*16,y,
       32*cd_sm,8*cd_sm)
 else --go!
  sspr(48,24,16,8,
      64-cd_sm*8,y,
      16*cd_sm,8*cd_sm)
 end
end


function draw_scoreboard()
 prn("congratulations!",32,15,9)
 
 --level times
 for i=0,lvls-1 do
  y = 40+i*12
 
  --level number, coloured like
  --the tile in the level
  prn("lvl "..i+1,4,y,m_tc[i+1][2])
  --level name
  prn("'"..m_name[i+1].."'",32,y,7)
  --level time
  --if you beat my time :)
  col = 6
  if (m_hs[i+1] < m_dhs[i+1]) col = 10
  prn(m_hs[i+1],76,y,col)
 end
 
 prn("press z/x to continue",22,118,7)
end


function _draw()
 --reset
 pal()
 rectfill(0,0,127,127,0)
 
 --main menu
 if game_state == gs_boot then
  if (t<=l_len) draw_logo()
  
 --title screen
 elseif game_state == gs_title then
  draw_title()
  
 --ingame
 elseif game_state == gs_game then
  --bg 
  draw_bg()
 
  --board
  draw_iso_map()
 
  --player
  draw_player()
 
  --in game menu
  if (g_paused) draw_g_paused()
 
  --g_timer
  draw_timer()
  
  --countdown
  if (t<=2*cntdwn) draw_countdown()
 
 --scoreboard
 elseif game_state == gs_score then
  draw_scoreboard()
 end
 
 if (tr_go) draw_transition()
 
 if (debug) print("cpu: "..stat(1),2,10,7)
end


-------------------
--update functions

--update if ingame
function update_game() 
 ------
 --live
 if not g_paused and
    t > cntdwn then

  --update g_timer
  g_timer += 1
  
  -----------------------
  ----inputs for movement 
  ---jump
  if c_z != 0 then
   c_z += c_vz
   c_vz -= c_g
  end
  if btn(4) and c_z==0 then 
   c_vz = c_av
   c_z += c_vz
   sfx(6,sd_jump)
   g_flp = not g_flp
  end
  if (c_z < 0) then
   c_z=0
   sfx(7,sd_jump)
  end
  
  ---cardinal movement
  --get acc for ground or air
  acc = c_acc
  deacc = c_deacc
  if c_z~=0 then 
   acc = c_aacc
   deacc = c_adeacc
  end
  
  --add acc to velocity
  if btn(1) then c_vx += acc
  elseif btn(0) then c_vx -= acc
  else c_vx *= deacc
  end
  if btn(3) then c_vy += acc
  elseif btn(2) then c_vy -= acc
  else c_vy *= deacc
  end
 
  --apply velocity
  if (abs(c_vx) > c_vmax) c_vx = sign(c_vx)*c_vmax
  if (abs(c_vy) > c_vmax) c_vy = sign(c_vy)*c_vmax
  c_x += c_vx
  c_y += c_vy
  
  --play rolling sound effect
  if (c_z==0) then
   vel = abs(c_vx)+abs(c_vy)
   if vel > c_vmax*1.4  then
    sfx(4,sd_roll)
   elseif vel > c_vmax*0.7  then
    sfx(0,sd_roll)
   elseif vel > c_vmax*0.1 then
    sfx(1,sd_roll)
   end
  end
 
--------------
--menu control
 elseif g_paused then
  --move menu cursor
  if (btnp(0)) p_c += 1
  if (btnp(1)) p_c -= 1
  if btnp(0) or btnp(1) and
      not g_lvl_death then
   sfx(10,sd_menu)
  end
  
  p_c = p_c%2
  
  --death menu
  --just reset
  if g_lvl_death then
   p_c = 0
   if btnp(4) then
    reset_level()
   end
  
  --level complete menu
  --next level and reset
  elseif g_lvl_end then
   if btnp(4) and p_c == 0 then
    tr_go = true
    --if beat last level,
    --go to scoreboard
    if lvl == lvls-1 then
     tr_to = tr_tos
    --otherwise go to next lvl
    else
     tr_to = tr_tog
    end
   end
   if btnp(4) and p_c == 1 then
    reset_level()
   end
  
  --user controlled pause menu
  --reseume and reset
  else
   if btnp(4) and p_c == 0 then
    g_paused = false
    sfx(9,sd_menu)
   elseif btnp(4) and p_c == 1 then
    reset_level()
   end
  end
 end
 
 --activate pause
 if btnp(5) and not g_paused and
    t > cntdwn then
  g_paused = true
  p_c = 0
  sfx(8,sd_menu)
 end
 
 ------------------------
 --check tile iteractions
 
 --update isometric pos of ball
 c_mx = flr((c_x + c_y)/sw) + m_x
 c_my = flr((c_y - c_x)/sw)
 
 --current tile type
 til = mget(c_mx,c_my)
 
 --if offmap
 if ((c_mx < m_x or
     c_mx > m_x + m_w - 1 or
     til == 0 or
     (til == 5 and g_flp) or
     (til == 21 and not g_flp)) and
     c_z == 0 and
     not g_paused) then
  g_lvl_death = true
  g_paused = true
  sfx(11,sd_menu)
 
 --if at map finish
 elseif (til == 17 and
         c_z == 0 and
         not g_lvl_end and
         not g_lvl_death) then
  music(-1)
  music(0,0,sd_music)
  c_vy = 0
  c_vx = 0
  c_vz = 0
  g_lvl_end = true
  g_paused = true
  if round(g_timer/30,2) < m_hs[lvl+1] then
   m_hs[lvl+1] = round(g_timer/30,2)
   m_nhs = true
  end
 end
end

function update_transition()
 tr_t += 1
 
 --if at pause point
 if tr_t == tr_l + 1 then
  --transition to game
  if tr_to == tr_tog then
   if game_state == gs_title then
    load_level(0)
   else
    load_level(lvl + 1)
   end
  --transition to scoreboard
  elseif tr_to == tr_tos then
   load_scoreboard()
  --transition to menu
  elseif tr_to == tr_tot then
   load_title()
  end
  
 --if transition has ended
 elseif tr_t == tr_l*2 + tr_p then
  tr_go = false
  tr_t = 0
  tr_to = -1
 end
end


function _update()
 --bootscreen
 if game_state == gs_boot then
  --if logo and wait has ended
  --go to title screen
  if t == l_len + l_wait then
   load_title()
  end
   
 --ttlescreen
 elseif game_state == gs_title then
  if (btn(4) or btn(5))
     and not tr_go then
   music(-1,tr_l*30)
   sfx(9,0)
   tr_go = true
   tr_to = tr_tog
  end
 
 --ingame
 elseif game_state == gs_game and
        not tr_go then
  update_game()
 
 --scoreboard
 elseif game_state == gs_score and
        not tr_go then
  if btnp(4) or btnp(5) then
   tr_go = true
   tr_to = tr_tot
   music(-1,tr_l*30)
   sfx(9,0)
  end
 end
 
 --transiion update
 if (tr_go) update_transition()
 
 --frame counters
 if not tr_go then
  t += 1 --stage frame
  tg += 1 --global frame
 end
end
__gfx__
0000000000000000000000000011110000000000000000000000000000000000000b000000777070770007070707077077070000000000000000000000000000
000000000bbbbbb00000000011223311000000000cccccc0000000000000000000bb000000707070707007070707070070070000000000000000000000000000
000000000bbbbbb00000001122332233110000000cccccc0000000000000000000bb888000707070707007070707070070070000000000000000000000000000
000000000bbbbbb00000112233223322331100000cccccc0000000000000000000bb888800707070707007070707070070070000000000000000000000000000
000000000bbbbbb00011223322332233223311000cccccc00000000000000000aaaacc0000777070707007070777077077070000000000000000000000000000
000000000bbbbbb01122332233223322332233110cccccc000000000000000000aaacc0000700070707007070707070070070000000000000000000000000000
000000000bbbbbb01133223322332233223322110cccccc000000000000000000000cc0000700070707007070707070070070000000000000000000000000000
00000000000000000011332233223322332211000000000000000000000000000000c00000700070707007070707070070070000000000000000000000000000
00000000000000000000113322332233221100000000000000000000000000000004900000700070707007770707070070070000000000000000000000000000
00000000077766600000001133223322110000000888888000000000000000000004900000700070707007070707077077077000000000000000000000000000
00000000077766600000000011332211000000000888888000000000000000000004900000000000000000000000000000000000000000000000000000000000
00000000077766600000000000111100000000000888888000000000000000000004900000006666066660660606660666600000000000000000000000000000
0000000006667770fff0f0f000fff000000000000888888000000000000000000004900000006600066060666606600660000000000000000000000000000000
00000000066677700f00f0f000ff0000000000000888888000000000000000000004900000006600066660660606660666600000000000000000000000000000
00000000066677700f00f0f000f00000000000000888888000000000000000000004900000006606066060660606600006600000000000000000000000000000
00000000000000000f00f0fff0fff000000000000000000000000000000000000000000000006666066060660606660666600000000000000000000000000000
008888000000000000000000000000000000000000000000aaaa00aaaa00aaa00aaaa00aa0a0aaaa000000000000000000000000000000000000000000000000
08ff88800000000000000000000000000000000000000000aa9aa0aa990aa9aa0aa9aa0aa0a0aa9a000000000000000000000000000000000000000000000000
8f88888800a0000000000000000a00000011110001111110aa0aa0aa000aa0aa0aa0aa0aa0a0aa0a000000000000000000000000000000000000000000000000
888888880000000000a0a000000000000111111011111111aaaaa0aaaa0aaaaa0aa0aa0aaaa0990a000000000000000000000000000000000000000000000000
8888888800000a000000000000a000000011110001111110aa9aa0aa990aa9aa0aa0aa09aa900aaa000000000000000000000000000000000000000000000000
8888888800a00000000a00000000a0000000000000000000aa0aa0aa000aa0aa0aa0aa00aa000999000000000000000000000000000000000000000000000000
088888800000000000000000000000000000000000000000aa0aa0aaaa0aa0aa0aaaa900aa000aa0000000000000000000000000000000000000000000000000
00888800000000000000000000000000000000000000000099099099990990990999900099000990000000000000000000000000000000000000000000000000
3333bbbb0770770007700777007770007700700070000777bbbbbb0bbbbb00bb0000000000000000000000000000000000000000000000000000000000000000
000000007557557075570755707557075570700070007555bb33330bb33bb0bb0000000000000000000000000000000000000000000000000000000000000000
dddd66667007007070070700707007070070700070007000bb00000bb00bb0bb0000000000000000000000000000000000000000000000000000000000000000
000000007007007077770777507775077770700070005770bb0bbb0bb00bb0bb0000000000000000000000000000000000000000000000000000000000000000
4444ffff7007007075570755707557075570700070000557bb03bb0bb00bb0bb0000000000000000000000000000000000000000000000000000000000000000
000000007007007070070700707007070070700070000007bb00bb0bb00bb0330000000000000000000000000000000000000000000000000000000000000000
eeee22227007007070070700707777070070777077707775bbbbbb0bbbbbb0bb0000000000000000000000000000000000000000000000000000000000000000
00000000500500505005050050555505005055505550555033333303333330330000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
00000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffff
__map__
0101010000000000000000000000000000010101010101010000000000000000010101000000000000000000000000000000000000000000000000000011110001010101010100000000000000000000010101010101010101010101010101010000000000000000000000000000000000000000000000000000000000000000
0101010100000000000000000000000001010101010101010100000000000000010101010000000000000000000000000000000000000000000000000011110001010101010100000000000000000000010101010101010101010101010101010000000000000000000000000000000000000000000000000000000000000000
0101010101000000000000000000000001010101010101010100000000000000010101010101010101010001010101010100000000000000000000000005050001010000010100000000000000000000010101000000000001010000000101010000000000000000000000000000000000000000000000000000000000000000
0001010101010000000000000000000001010101000101010100000000000000000101010101010101010001010101010101000000000000000000000005050001010000010100000000000000000000010101000000000001010000000101010000000000000000000000000000000000000000000000000000000000000000
0000010101010100000000000000000001010100000001010100000000000000000001010101010101010001010101010101000000000000000000000000000001010101010100000000000000151515050505000000000001010100000101010000000000000000000000000000000000000000000000000000000000000000
0000000101010101000000000000000001010100000001010100000000000000000000000000000000000000000000010101000000000000000000000005050001010101010100000000000000151515050505000000000001010101010101010000000000000000000000000000000000000000000000000000000000000000
0000000001010101010000000000000001010100000001010100000000000000000000000000000000000000000000010101000000000000000000000005050001010005050000000000000000151515050505000000000001010101010101010000000000000000000000000000000000000000000000000000000000000000
0000000000010101010100000000000001010100000001010100000000000000000000000000000000000000000000010101000000000000000000000005050000000005050000000000000000050505151515000000000000000000000001010000000000000000000000000000000000000000000000000000000000000000
0000000000000101010101010000000101010100000001010100000000000000000000000000000000000000000000010101000000000000000000000000000000000001010000000000000000050505151515000000000000000000000001010000000000000000000000000000000000000000000000000000000000000000
0000000000000001010101010101010101010100000001010100000000000000010101000000010101010000010101010101000000000000000000000000000000000001010000000000000000050505151515000000000000000000000101010000000000000000000000000000000000000000000000000000000000000000
0000000000000000010101010101010101010000000001010100000000000000010101000000010101010000010101010101000000000000000000000001010000000001010000000000000000151515050505000000000000000000010101010000000000000000000000000000000000000000000000000000000000000000
0000000000000000000001010101010101000000000001010100000000000000010101000000010101010000010101010100000000000000000000000001010000000001010000000000000000151515050505000000000000000001010101000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000101010100000000000000010101000000000000000000000000000000000000000000000000000001010000000001010000000000000000151515050505000000000000000101010100000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000010101010100000000000000010101000000000000000000000000000000000000000000000000000001010000000001010000000000000000010101010101000000000000000101010000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000001010101010000000000000000010101000000000000000000000000000000000000000000000000000005050000000001010000000000000000010101010101000000000000000101000000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000101010101000000000000000000010101000000000000000000000000000000000000151515150505151505051500000000000000000000000000000001010000000000000000000101000000000000000000000000000000000000000000000000000000000000000000000000
0000000101010100000000000000000101010101010100000000000000000000010101010000000000000000000000000000000000151515150505151505051500000000000000000000000101000001010000000000000000000101000000000000000000000000000000000000000000000000000000000000000000000000
0001010101010101010000000001010101010101010000000000000000000000010101010100000000000000000000000000000000151500000000000000000001010101000000000000000101000001010000000000000000000101000000000000000000000000000000000000000000000000000000000000000000000000
0001010101010101010100000101010101010100000000000000000000000000000101010101000000000000000000000000000000150500000000000000000001010101000000000000000101000001010000000000000000010101010000000000000000000000000000000000000000000000000000000000000000000000
0101010100000101010101010101010100000000000000000000000000000000000001010100000000000000000000000000000000050500000000000000000001010101000000000000000101000001010000000000000000010101010000000000000000000000000000000000000000000000000000000000000000000000
0101010000000000010101010101000000000000000000000000000000000000000000010000000100000000000000000000000000050500000000000000000001011515000000000000000000000000000000000000000001010101010100000000000000000000000000000000000000000000000000000000000000000000
0101010000000000000101010100000000000000000000000000000000000000000000000000010101000000000000000000000000050500000000000000000001011515000000000000000000000000000000000000000015150000050500000000000000000000000000000000000000000000000000000000000000000000
0101010000000000000000000000000000000000000000000000000000000000000000000001010101010000000000000000000000051500000000000000000001010101000000000000000505010101010000000000001515150000050505000000000000000000000000000000000000000000000000000000000000000000
0101010100000000000000000000000000000000000000000000000000000000000000000000010101010100000000000000000000151515151500010101010101010101000000000000000505010101010000000000000101000000000101000000000000000000000000000000000000000000000000000000000000000000
0001010101010000000000000000010101010101010101010101010101010101000000000000000101010101000000000000000000151515151500010101010101010101000000000000000505000001010000000000000101000000000101000000000000000000000000000000000000000000000000000000000000000000
0000010101010101010101010101010101010101010100010101010101010101000000000000010101010100000000000000000000000000000000000000010100000000000000000000000505000001010000000000000505000000001515000000000000000000000000000000000000000000000000000000000000000000
0000000101010101010101010101010101010001010100010101000101010101000000000001010101010000000000000000000000000005150000000000010100000101151515150000000505010101010000000000000505000000001515000000000000000000000000000000000000000000000000000000000000000000
0000000000010101010101010101010101010001010101010101000101010101000000000001010101000000000000000000000000000015050000000000051500000101151515150000000505010101010000000000000101000000000101000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000010101010101010101010101000101010101000000000001010100000000000005001500050015000005150000000000010100000101000015150000000505000001010000000000000101010000010101000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000010101000000000000000000000000000000000000000000000000000000000000010100000101000015150000000505000001010000000000000001010101010100000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000111111000000000001010101010101010100010100010100000101010001010001010100000101010101010000000505010101010000000000000000010101010000000000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000111111000000000001010101010101010100010100010100000101010001010001010100000101010101010000000505010101010000000000000000001111000000000000000000000000000000000000000000000000000000000000000000000000
__sfx__
00050000146200e6110160001600016000160001600016000960005600086000c6000a60008600056000360002600016000160001600016000160001600016000160001600016000160001600016000160001600
0104000002610016110760001600016000160001600016000360003600036000360001600016000160003600036000360004600046001f6000460004600046000460004600046000360001600016001f6001f600
010e00001a4701a4711a4611a4511a4411a4311c4701c4611f4701f4611f4511e4701e4611e4511c4701c46121470214712146121451214412143121421214112140000400004000040000400004000040000400
010e00000707007061070510704107031070210701107011070700706107051070410703107021070110701102070020610205102041020310202102011020000700007000070000700007000000000000006000
01070000176301a611186000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
010e0000110432460224603110433064524603006452460511043246052460511043376452460605643246073e6402d651236511b64115631116210a611076000560004600036050360502605026050260502605
010400000c0600e051100411103113021150211702118000244000d4000e4000e4000e4002540008400084001c406214061c406214061c406214061c406214060340503405034050340503405034050340503405
010e00001504300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
010500001c346213461c346213461c3401c3001c306213061c306213061c306213061a30621306000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
010500001a346213461a346213461a346213402130000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
000d00000e3251a145001000010000100001000010000100001000010000100001000010000100001000010000100001000010000100001000010000100001000010000100001000010000100001000010000100
00050000211401f1411d1411c1411a141181411a1411c1411a1411814117141151410c100000000000000000000000000000000000000000000000000003b1000000000000000000000000000000000000000000
00150000170751e07523075170751e07521075170751e075260751207519075250751207519075210752307515075190752107515075190752007515075190751c0751507519075210751c0751e0752107520075
001500000b1550b1550b1550b1550b1550b1550b1550b155061550615506155061550615506155061550615509155091550915509155091550915509155091550915509155091550915509155091550915509155
01150000150730220539613150732e655396033961315003150731500315073150732e655000002d61315003150731500339613150732e655150032d61315003150031507315653150732e655000002864528655
011500001735217352173421734217332173321732217322123521235212342123421233212332123221232215352153521534215342153321533215322153221532215312153121531215302153021530215302
011500001e3521e3521e3421e3421e3321e3321e3221e322193521935219342193421933219332193221932219352193521934219342193321933219322193221932219312193121931219302193021930219302
011500002636226352263422633225362253522534225332213622135221342213322336223352233422333220362203522034220342203322033220322203222032220312203122031220302203022030220302
014100101a1440e2440e3541a4541a1440e2440c354184541d14411244113541d45420144142441435422454222042e0042e00030000107032e0042e00030000107032e0042e00030000107032e0042e00030000
014100100e45402354022440e4440e35402254004440c35411254054440534411254144540834408244163342220022202222022220222202222022220222202212002120221202212052620226202262051a202
014100100205502055020550205502055020550005500055050550505505055050550805508055080550a05502405024050e4050e40502305023050230502305004050040500405004050c3050c3050c3050c305
011000001141011421114311144105310053210533105341054100542105431054411131011321113311134114410144211443114441083100832108331083410841008421084310844116310163211633116341
012e0000110451504516045180551c0551d0552106522065110451504516045180551c0551d0552106522065110451504516045180551c0551d0552106522065110451504516045180551c0551d0552106522065
012e00000512005121051210512105111051110511105115051200512105121051210511105111051110511505120051210512105121051110511105111051150512005121051210512105111051110511105115
012e00001d54521545225452455528555295552d5652e5651d54521545225452455528555295552d5652e5651d54521545225452455528555295552d5652e5651d54521545225452455528555295552d5652e565
012e00000c04510045110451305516055180551c0651d0650c04510045110451305516055180551c0651d06522065210651d0651c0551805515055160451804522065210651d0651c05518055150551604518045
012e0000185451c5451d5451f55522555245552856529565185451c5451d5451f555225552455528565295652e5652d5652956528555245552155522545245452e5652d565295652855524555215552254524545
012e0000001200012100121001210011100111001110011500120001210012100121001110011100111001150a1200a1210a1210a1210a1110a1110a1110a1150a1200a1210a1210a1210a1110a1110a1110a115
012e000005035050001102511005050350500511025110050503509005110250c00505035050050c025160050503509005110250c005050350500511025160050503509005110250c00505035050050c02507005
012e000000035151050c02518105000351d1050c02522105000350c0050c02500100000350010010025001000a0350000516025001000a0350010011025001000a0350010016025001000a035001001502500100
010e00201824018261242622426224200242002426024265302603026030200302001824018261242622426209200242002426024265302603026009200092001f2401f2612b2602b26222240222612e2622e262
011c0000000350013500135000350013500135000350013500135000350013500135000350013500135000350a0350a1350a1350a0350a1350a1350a0350a1350a1350a0350a1350a1350a0350a1350a1350a035
011c00000c4250f43513445184450c4250f43513445184450c4250f43513445184450c4250f43513445184450a4250e43511445164450a4250e43511445164450a4250e43511445164450a4250e4351144516445
011c0000054250c4351144514445054250c4351144514445054250c4351144514445054250c4351144514445084250c4350f44514445084250c4350f44514445084250c4350f44514445084250c4350f44514445
011c00000503505135051350503505135051350503505135051350503505135051350503505135051350503508035081350813508035081350813508035081350813508035081350813508035081350813508035
010700103c6253c600246151c4003c63024600246151a4003065024651186310c6213c63024600246201b61526400264002540025400254002540025400254002140021400214002140021400214001e4001e400
011c00003c6503064124641186310c6210c6110c6000000000000000000000000000000000000000000000001540015400154001540015400154001540015400214002140021400214001e4001e4001e4001e400
010e00001307302100001500015013073021000015000150130730e10000150001501307302100001500015013073021000815008150130730e100081500815013073021000a1500a150130730e1000a1500a150
010e00001307300100001500015013073001000015000150130730000000150001501307300000001500015013073000000015000150130730000000150001501307300000001500015013073000000015000150
0107001039600396023960339604396053960639600306003065024651186310c621006110c6000c6002460032200322003120031200312003120031200312002d2002d2002d2002d2002d2002d2002a2002a200
011c00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001620016200162001621116221162311624116261
010e00000011000131001410015100110001310014100151001100013100141001510011000131001410015108110081310814108151081100813108141081510a1100a1310a1410a1510a1100a1310a1410a151
011c00000c4250f43513445184450c4250f43513445184450c4250f43513445184450c4250f43513445184450a4250e43511445164450a4250e43511445164451426014260082600a26016270162700027000270
010e0000002100023100241002510021000231002410025100210002310024100251002100023100241002510021000231002410025100210002310024100251002100023100241002510c2100c2310c2410c251
010e0000002100023100241002510021000231002410025100210002310024100251002100023100241002511421014231142411425108210082310a2400a251162101623116241162510c2100c2310c2410c251
010e00001307300010000210002013073000100002100020130730001000021000201307300010000210002013073000100002100020130730001000021000201307300010000210002013073000100002100020
010e0000130730001000021000201307300010000210002013073000100002100020130730001000021000201307308010080210802013073080100a0200a020130730a0100a0210a02013073000100002100020
010d0000155601556516500155601556500500185601856518500185601856500500185601856518560185651856018565005001856018565005001b5601b565005001b5601b5650050017560175651656016565
010d00001656016565005001656016565005001156011565005001156011565005001156011565145601456513560135650050013560135650050016560165650050016560165650050016560165651856018565
010d00001d5301d535295001d5301d535005001d5301d535005001d5301d535005001d5301d5351d5301d5351d5301d5351d5051d5301d5351d5051d5301d5351d5051d5301d5351d5051d5301d5351d5301d535
010d00101d5301d535000002453024535000001d5301d5351400024530245351d5051d5301d53524530245351d5051d535245351d5051d535245351d5051d535245351d5051d535245351d5051d535245351d535
010d000009240092400d200092400924000000092400924000000092400924000000092400924009240092000b2400b240000000b2400b240000000b2400b240000000b2400b240000000b2400b2400b24000000
010d00001857018571185711856118561185611855118551185511854118541185411853118531185311852118521185211851118511185111850018500185000000000000000000000000000000000000000000
010d00003962500000000002d6052d6250000000000000002d6250000000000000002d6150000000000000002d6150000000000000002d6150000000000000002d6050000000000000002d605000000000000000
010d00002135021341213312132121311213002235022341223312232122311223002935029341293312932129311293002035020341203312032120311203000000000000000000000000000000000000000000
010d00002035020341203312032120311203002235022341223312232122311223002035020341203312032120311203001f3501f3411f3311f3211f3111f3001d3501d3411d3311d3211f3501f3411f3311f321
010d00002135021341213312132121311213002235022341223312232122311223002035020341203312032120311203001f3501f3411f3311f3211f3111f3000000000000000000000019300193001930019300
010d00001935019341193311932119311193001b3501b3411b3311b3211b3111b3002235022341223312232122311223002435024341243312432124311243002135021341213312132121311213000000000000
010d00000513005130051300513005132051320513205132151301513015130151301513215132151321513208130081300813008130081320813208132081321613016130161301613016132161321613216132
010d00000113001130011300113001132011320113201132141301413014130141301413214132141321413203130031300313003130031320313203132031321313013130131301313013132131321313213132
011200000c53500500165350050018535005001850518535185350f535005000f535005000f5350f535005000f5350f505115350050011535005000050011535135350a535005000a535005000a5350753500500
011200000c0300c0310c0310c0310c0210c0210c0210c0210c0110c0110c0110c0110c0000c0000c0000c0000f0300f0310f0310f0310f0210f0210f0210f0210a0300a0310a0310a0310a0210a0210a0210a021
__music__
04 42030502
00 4c4d0c4e
00 4c4d0c0d
01 410e0c0d
00 410e0c0d
00 410e0c0d
00 410f1011
00 410c0d11
00 410c0d10
00 410d0c0f
02 410f1011
01 171c1618
00 1b1d191a
00 1c171618
02 1d1b191a
01 41541312
02 41131412
00 24424344
03 23252667
01 63652420
00 41422821
00 671f1e20
00 41221e21
00 415f1e26
00 41671e26
00 41271e26
00 41271e29
00 411e2623
00 411e2523
00 411e2623
00 411e2923
00 4142242a
00 412d2b1e
00 412e2c1e
00 412d2b1e
02 412e2c1e
00 416b6d72
00 412c2e32
00 412b2f32
00 412c3032
00 41347549
00 413a366d
00 413b3771
00 413a3832
02 413b3932
00 412b2d36
00 412c2e37
00 412b2f36
02 412c3039
00 41424344
00 41424344
00 417c7d44

